// @license magnet:?xt=urn:btih:0b31508aeb0634b347b8270c7bee4d411b5d4109&dn=agpl-3.0.txt AGPL-3.0
/*global*/
/*eslint no-undef: "error"*/



/**
* SERVICE WORKER
* Manages Dédalo files cache when is fired by login, acting as proxy.
* Once registered, it handles all requests, being transparent
* for requests for files that are not in the cached list.
* The registration is done by the login, after authorizing the user.
* Note that files_list is generated by API dd_utils_api::get_dedalo_files
* and is a list of main javascript Dédalo core and tools modules cached
* with this service to take controls about files changes on every user login.
* On user logout (quit), the service worker is unregistered and cache is deleted.
* This is HTTPS only service !
*/



/**
* DATA_MANGER REQUEST CUSTOM
* Avoid to use data_manager module to allow happy Firefox users
*/
const data_manager = {
	request : async function(options) {

		// options
			this.url			= options.url
			this.method			= options.method || 'POST' // *GET, POST, PUT, DELETE, etc.
			this.mode			= options.mode || 'cors' // no-cors, cors, *same-origin
			this.cache			= options.cache || 'no-cache' // *default, no-cache, reload, force-cache, only-if-cached
			this.credentials	= options.credentials || 'same-origin' // include, *same-origin, omit
			this.headers		= options.headers || {'Content-Type': 'application/json'}// 'Content-Type': 'application/x-www-form-urlencoded'
			this.redirect		= options.redirect || 'follow' // manual, *follow, error
			this.referrer		= options.referrer || 'no-referrer' // no-referrer, *client
			this.body			= options.body // body data type must match "Content-Type" header

		// handle_errors
			const handle_errors = function(response) {
				if (!response.ok) {
					console.warn("-> HANDLE_ERRORS response:",response);
					throw Error(response.statusText);
				}
				return response;
			}

		const api_response = fetch(
			this.url,
			{
				method		: this.method,
				mode		: this.mode,
				cache		: this.cache,
				credentials	: this.credentials,
				headers		: this.headers,
				redirect	: this.redirect,
				referrer	: this.referrer,
				body		: JSON.stringify(this.body)
			})
			.then(handle_errors)
			.then(response => {
				const json_parsed = response.json().then((result)=>{

					if (result.error) {

						// debug console message
							console.error("result error:",result);

						// alert msg to user
							const msg = result.msg || result.error
							console.error("An error occurred in the connection with the API (worker cache data_manager). \n" + msg);

						// custom behaviors
							switch (result.error) {
								case 'not_logged':
									// redirect to login page
									// location.reload();
									console.warn('Result error. no logged!', result);
									break;

								default:
									// write message to the console
									break;
							}
					}

					return result
				})

				return json_parsed
			})
			.catch(error => {
				console.error("!!!!! [data_manager.request] SERVER ERROR. Received data is not JSON valid. See your server log for details. catch ERROR:\n", error)
				console.warn("options:", options);
				return {
					result	: false,
					msg		: error.message,
					error	: error
				}
			});


		return api_response
	}
}//end data_manager



/**
* @vars
*/
// string cache_name. Used to identify current cache. Normally 'dedalo_files'
const cache_name = 'dedalo_files'
// array files_list. Filled with the list of Dédalo main javascript files
let files_list



/**
* LOAD_FILES_LIST
* Call API to get a list of the Dédalo main javascript files
* to cache and fix the value in 'files_list' var
* @return array|null files_list
*/
const load_files_list = async () => {

	if (files_list) {
		return files_list
	}

	// get_dedalo_files from API
		const api_response = await data_manager.request({
			url		: './api/v1/json/', // DEDALO_API_URL,
			body	: {
				action	: 'get_dedalo_files',
				dd_api	: 'dd_utils_api'
			}
		});

	// API error case
		if (!api_response.result) {
			console.error('Error on get api response:', api_response);
			self.postMessage({
				status	: 'finish',
				error	: 'Error on get api response'
			});
			return null
		}

	// fix values
		files_list = api_response.result.map(el => el.url)


	return files_list
}//end load_files_list




/**
* ADD_RESOURCES_TO_CACHE
* Load file list from API
* Put files_list in cache sending a message for each file load
* @param object event
* @return void
*/
const add_resources_to_cache = async (options) => {

	// options unpack
	const {
		load_file_handler
	} = options

	// fire load files from API
	await load_files_list()

	// open cache interface
	const cache = await caches.open( cache_name );

	// headers constructor for js files
	const get_headers = () => {
		const headers = new Headers();
		// time: one week = 604800 (7 x 24 x 60 x 60)
		headers.append('Cache-Control', 'stale-while-revalidate=604800'); // no-cache
		// mime: text/javascript
		headers.append('Content-Type', 'text/javascript');
		return headers
	}

	// custom cache add equivalent to 'cache.add' but using fetch param 'cache' as 'reload'
	// to force load the file from server always
	const custom_cache_add = async (url) => {
		const response = await fetch(url, {
			headers	: get_headers(),
			method	: 'GET',
			cache	: 'reload'
		})
		if (!response.ok) {
			throw new TypeError("bad response status");
		}
		return cache.put(url, response);
	}

	// await cache.addAll( files_list );
	// add files one by one to allow post messages
	const ar_promises = []
	const files_list_length = files_list.length
	for (let i = 0; i < files_list_length; i++) {

		// const ad_promise = cache.add( files_list[i] );
		const ad_promise = custom_cache_add( files_list[i] );

		ar_promises.push(ad_promise)

		// notify file is loaded into cache (for files loader circle using loading handler)
		ad_promise.then(()=>{
			load_file_handler(i, files_list_length)
		})
	}
	await Promise.all(ar_promises)
}//end add_resources_to_cache



/**
* DELETE_CACHE
* Delete caches item by key
* @param object key
* @return void
*/
const delete_cache = async (key) => {
	await caches.delete(key);
}//end delete_cache



/**
* DELETE_OLD_CACHES
* Delete whole caches except the given name
* @param string cache_name
* @return void
*/
const delete_old_caches = async (cache_name) => {

	console.log(')) deleting cache:', cache_name);

	const cacheKeepList		= [cache_name];
	const keyList			= await caches.keys();
	const cachesToDelete	= keyList.filter((key) => !cacheKeepList.includes(key));
	await Promise.all(cachesToDelete.map(delete_cache));
}//end delete_old_caches



/**
* CACHE_FIRST
* Check for request file from cache. If file is found, return from cache,
* else request the file normally
* @param object request
* @return promise
* 	fetch promise
*/
const cache_first = async (request) => {
	const responseFromCache = await caches.match(request);
	if (responseFromCache) {
		return responseFromCache;
	}
	return fetch(request);
}//end cache_first



/**
* INSTALL EVENT
*/
self.addEventListener('install', async (event) => {

	console.log(')) install event:', event);
});



/**
* ACTIVATE EVENT
*/
self.addEventListener('activate', async (event) => {

	console.log(')) activate event:', event);
});



/**
* MESSAGE EVENT
*/
self.addEventListener('message', async (event) => {

	console.log(')) message event:', event.data, event);

	if (event.data==='update_files') {
		const t1 = performance.now()

		// ready message (enable loading files circle render by ready handler)
			event.source.postMessage({
				status : 'ready'
			});

		// load files updating existing ones
			const load_file_handler = (key, total_files) => {
				// on_load_file
				event.source.postMessage({
					status		: 'loading',
					key			: key,
					total_files	: total_files,
					file_loaded	: true
				})
			}
			add_resources_to_cache({
				load_file_handler : load_file_handler
			})
			.then(()=>{
				// all files are loaded. Notify finish
				event.source.postMessage({
					status		: 'finish',
					total_files	: files_list.length,
					time		: performance.now()-t1
				});
			})
	}
});



/**
* FETCH EVENT
*/
self.addEventListener('fetch', (event) => {

	// debug
		// console.log('event.request.url:', event.request.url);

	// Manage only javascript files excluding images, css, etc.
	// that will be handled by the browser normally
	if (	event.request.url.endsWith('.js')
		&& !event.request.url.endsWith('/sw.js')
		&& !event.request.url.includes('/lib/')
		) {

		event.respondWith(
			cache_first(event.request)
		);
	}
});



// @license-end
